<?php
session_start();
require_once '../config/db.php';
require_once '../config/config.php';

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '../logs/mpesa_invoice_errors.log');

// Create logs directory if it doesn't exist
if (!file_exists('../logs')) {
    mkdir('../logs', 0777, true);
}

// Log the callback for debugging
$logFile = '../logs/mpesa_invoice_callback.log';
try {
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Invoice callback received\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("Failed to write to log file: " . $e->getMessage());
}

// Get the callback data
$rawData = file_get_contents('php://input');
if ($rawData === false) {
    error_log("Failed to read input data");
    exit();
}

$callbackData = json_decode($rawData, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    error_log("JSON decode error: " . json_last_error_msg());
    exit();
}

try {
    file_put_contents($logFile, "Callback data: " . json_encode($callbackData) . "\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("Failed to write callback data to log: " . $e->getMessage());
}

// Verify the callback
if (!isset($callbackData['Body']['stkCallback'])) {
    try {
        file_put_contents($logFile, "Invalid callback format\n", FILE_APPEND);
    } catch (Exception $e) {
        error_log("Failed to write invalid format message: " . $e->getMessage());
    }
    exit();
}

$stkCallback = $callbackData['Body']['stkCallback'];
$resultCode = $stkCallback['ResultCode'] ?? null;
$resultDesc = $stkCallback['ResultDesc'] ?? 'Unknown error';
$merchantRequestID = $stkCallback['MerchantRequestID'] ?? '';
$checkoutRequestID = $stkCallback['CheckoutRequestID'] ?? '';

// Log the result
try {
    file_put_contents($logFile, "Result Code: " . $resultCode . "\n", FILE_APPEND);
    file_put_contents($logFile, "Result Description: " . $resultDesc . "\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("Failed to write result to log: " . $e->getMessage());
}

if ($resultCode == 0) {
    // Payment was successful
    $callbackMetadata = $stkCallback['CallbackMetadata']['Item'] ?? [];
    
    // Extract payment details
    $amount = 0;
    $mpesaReceiptNumber = '';
    $transactionDate = '';
    $phoneNumber = '';
    
    foreach ($callbackMetadata as $item) {
        switch ($item['Name']) {
            case 'Amount':
                $amount = $item['Value'] ?? 0;
                break;
            case 'MpesaReceiptNumber':
                $mpesaReceiptNumber = $item['Value'] ?? '';
                break;
            case 'TransactionDate':
                $transactionDate = $item['Value'] ?? '';
                break;
            case 'PhoneNumber':
                $phoneNumber = $item['Value'] ?? '';
                break;
        }
    }
    
    try {
        // Begin transaction
        $pdo->beginTransaction();
        
        // Update the payment status in your database
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET 
                status = 'completed',
                reference_number = ?,
                notes = CONCAT(notes, ' | M-Pesa Receipt: ', ?),
                updated_at = NOW()
            WHERE 
                reference_number LIKE ? 
                AND status = 'pending'
        ");
        
        $checkoutRef = 'MPESA-' . $checkoutRequestID;
        $stmt->execute([
            $mpesaReceiptNumber,
            $mpesaReceiptNumber,
            $checkoutRef
        ]);
        
        $updatedRows = $stmt->rowCount();
        
        // If no rows were affected, log it
        if ($updatedRows === 0) {
            error_log("No pending payment found for checkout_request_id: " . $checkoutRequestID);
        } else {
            file_put_contents($logFile, "Updated {$updatedRows} payment record(s) for M-Pesa receipt: {$mpesaReceiptNumber}\n", FILE_APPEND);
        }
        
        // Update invoice status based on total payments
        $stmt = $pdo->prepare("
            SELECT 
                p.invoice_id,
                i.total_amount,
                SUM(CASE WHEN p.status = 'completed' THEN p.amount ELSE 0 END) as total_paid
            FROM payments p
            JOIN invoices i ON p.invoice_id = i.id
            WHERE p.invoice_id = (
                SELECT invoice_id FROM payments 
                WHERE reference_number = ? AND status = 'completed'
                LIMIT 1
            )
            GROUP BY p.invoice_id
        ");
        
        $stmt->execute([$mpesaReceiptNumber]);
        $invoicePayment = $stmt->fetch();
        
        if ($invoicePayment) {
            if ($invoicePayment['total_paid'] >= $invoicePayment['total_amount']) {
                // Full payment received - mark as paid
                $stmt = $pdo->prepare("UPDATE invoices SET status = 'paid', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$invoicePayment['invoice_id']]);
                
                file_put_contents($logFile, "Invoice {$invoicePayment['invoice_id']} marked as paid (full payment)\n", FILE_APPEND);
            } else {
                // Partial payment - keep as pending
                $stmt = $pdo->prepare("UPDATE invoices SET status = 'pending', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$invoicePayment['invoice_id']]);
                
                $remainingBalance = $invoicePayment['total_amount'] - $invoicePayment['total_paid'];
                file_put_contents($logFile, "Invoice {$invoicePayment['invoice_id']} partial payment. Remaining: Ksh {$remainingBalance}\n", FILE_APPEND);
            }
        }
        
        // Update M-Pesa transaction record
        $stmt = $pdo->prepare("
            UPDATE mpesa_transactions 
            SET 
                mpesa_receipt = ?,
                transaction_date = ?,
                status = 'completed',
                updated_at = NOW()
            WHERE 
                checkout_request_id = ?
        ");
        
        $stmt->execute([
            $mpesaReceiptNumber,
            $transactionDate,
            $checkoutRequestID
        ]);
        
        $pdo->commit();
        file_put_contents($logFile, "Invoice payment processed successfully\n", FILE_APPEND);
        
        // Send email notification after successful payment
        try {
            // Get invoice and customer details
            $stmt = $pdo->prepare("
                SELECT 
                    i.id as invoice_id,
                    i.invoice_number,
                    i.total_amount,
                    c.name as customer_name, 
                    c.email, 
                    c.phone
                FROM invoices i
                JOIN customers c ON i.customer_id = c.id
                WHERE i.id = ?
                LIMIT 1
            ");
            $stmt->execute([$invoicePayment['invoice_id'] ?? null]);
            $invoiceDetails = $stmt->fetch();
            
            if ($invoiceDetails && !empty($invoiceDetails['email'])) {
                // Send payment confirmation email
                require_once '../includes/email_functions.php';
                require_once '../includes/email_templates.php';
                
                $subject = "Invoice Payment Confirmation - Receipt #" . $mpesaReceiptNumber;
                $emailBody = createInvoicePaymentConfirmationEmail(
                    $invoiceDetails['customer_name'],
                    $invoiceDetails['invoice_number'],
                    $amount,
                    $mpesaReceiptNumber,
                    $transactionDate
                );
                
                $emailSent = sendEmail($invoiceDetails['email'], $subject, $emailBody);
                
                if ($emailSent) {
                    file_put_contents($logFile, "Payment confirmation email sent to: " . $invoiceDetails['email'] . "\n", FILE_APPEND);
                } else {
                    file_put_contents($logFile, "Failed to send payment confirmation email to: " . $invoiceDetails['email'] . "\n", FILE_APPEND);
                }
            } else {
                file_put_contents($logFile, "No customer email found for invoice: " . ($invoicePayment['invoice_id'] ?? 'unknown') . "\n", FILE_APPEND);
            }
        } catch (Exception $emailError) {
            file_put_contents($logFile, "Error sending payment confirmation email: " . $emailError->getMessage() . "\n", FILE_APPEND);
        }
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Database error in invoice payment callback: " . $e->getMessage());
        file_put_contents($logFile, "Database error: " . $e->getMessage() . "\n", FILE_APPEND);
    }
    
} else {
    // Payment failed
    try {
        // Update payment status to failed
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET 
                status = 'failed',
                notes = CONCAT(notes, ' | M-Pesa Error: ', ?),
                updated_at = NOW()
            WHERE 
                reference_number LIKE ? 
                AND status = 'pending'
        ");
        
        $checkoutRef = 'MPESA-' . $checkoutRequestID;
        $stmt->execute([
            $resultDesc,
            $checkoutRef
        ]);
        
        // Update M-Pesa transaction record
        $stmt = $pdo->prepare("
            UPDATE mpesa_transactions 
            SET 
                status = 'failed',
                error_message = ?,
                updated_at = NOW()
            WHERE 
                checkout_request_id = ?
        ");
        
        $stmt->execute([
            $resultDesc,
            $checkoutRequestID
        ]);
        
        file_put_contents($logFile, "Payment failed: " . $resultDesc . "\n", FILE_APPEND);
        
    } catch (PDOException $e) {
        error_log("Database error updating failed payment: " . $e->getMessage());
        file_put_contents($logFile, "Database error updating failed payment: " . $e->getMessage() . "\n", FILE_APPEND);
    }
}

// Send response to M-Pesa
http_response_code(200);
echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Success']); 